# Copyright (c) 2018-2024 Broadcom. All Rights Reserved.
# Broadcom Confidential. The term "Broadcom" refers to Broadcom Inc.
# and/or its subsidiaries.

"""System Storage common definitions.
"""
import os

IS_ESX = 'VMkernel' in os.uname().sysname

if IS_ESX:
   # System storage feature state is not available on 6.x. In which case
   # set feature variable to False since not all capabilities, such as
   # userworld formatting of VFAT or system VMFS-L, are there.
   libssPath = os.path.join(os.path.sep, 'lib64', 'libsysstorage.so')
   IS_SYSTEM_STORAGE_NEXT_ENABLED = os.path.isfile(libssPath)
else:
   # Linux scripts which rely on this flag always assume SystemStorageNext
   IS_SYSTEM_STORAGE_NEXT_ENABLED = True

# Units
MiB = 1024 * 1024
GiB = 1024 * MiB

# Filesystem limits
VMFS_MIN_SIZE_MB = 2560
VMFSL_MIN_SIZE_MB = VMFS_MIN_SIZE_MB

# FAT16 hard limit; 128 sectors per cluster, 64k (minus reserved) clusters.
VFAT_MAX_SECTORS = 128 * 65524

# 512 bytes per sector for compatibility across different platforms
VFAT_MAX_PART_SIZE_MB = VFAT_MAX_SECTORS * 512 // MiB

# UEFI / boot partition size
SYSTEM_BOOT_SIZE_MB = 100

#
# Disk sizes are spec'd as 1000's by vendors. ESX will support
# media sizes of 4GB and up, which is around 3.8GiB (in base-2).
#
# These are the minimum disk size brackets for each corresponding bootbank size.
#
_VENDOR_GB = 1000 * 1000 * 1000
SYSTEM_DISK_SIZE_SMALL_MB =   4 * _VENDOR_GB // MiB
SYSTEM_DISK_MIN_HDD_SIZE_MB = 8 * _VENDOR_GB // MiB
SYSTEM_DISK_SIZE_MEDIUM_MB = 10 * _VENDOR_GB // MiB
SYSTEM_DISK_SIZE_LARGE_MB =  31 * _VENDOR_GB // MiB

#
# Bootbank sizes paired to the above minimum disk sizes.
# These are appropriately sized to enable OSData to fit on the media, as well
# as increasing bootbank sizes.
#
#  <  10GB media: small bootbank size of 500MiB each
#  >= 10GB media: small bootbank size of 1GiB each
#  >= 32GB media: large bootbank size utilizing max size of FAT16 partition
#
BOOTBANK_SIZE_SMALL_MB = 500
BOOTBANK_SIZE_MEDIUM_MB = 1024
BOOTBANK_SIZE_LARGE_MB = VFAT_MAX_PART_SIZE_MB

#
# Filesystem types
#
FS_TYPE_UEFI_SYSTEM = "uefi"
FS_TYPE_EXTENDED = "extended"
FS_TYPE_FAT16 = "fat16"
FS_TYPE_FAT32 = 'fat32'
FS_TYPE_VFAT = "vfat"
FS_TYPE_VMFS = "vmfs"
FS_TYPE_VMFS_L = "vmfsl"
FS_TYPE_VMFSOS = "vmfsos"
FS_TYPE_VMKCORE = "vmkcore"
FS_TYPE_DELL_UTILITY = "dell"
FS_TYPE_EFI_GPT = "efiGpt"
FS_TYPE_UNKNOWN = "unknown"

#
# System partition labels
#
BOOTPART_LABEL = 'BOOT'
BOOTBANK1_LABEL = 'BOOTBANK1'
BOOTBANK2_LABEL = 'BOOTBANK2'
OSDATA_LABEL = 'OSDATA'
LOCKER_LABEL = 'LOCKER'

#
# System partition filesystems
#
LEGACY_SCRATCH_FS = FS_TYPE_VFAT
LEGACY_LOCKER_FS = FS_TYPE_VFAT

BOOTPART_FS = FS_TYPE_UEFI_SYSTEM
BOOTBANK_FS = FS_TYPE_VFAT
OSDATA_FS = FS_TYPE_VMFS_L
LOCKER_FS = FS_TYPE_VMFS_L

# VMFS config bit setting for OS-Data volume (fs_public.h)
FS_CONFIG_SYSTEM = 2048

# VMFS createFlags for OS-Data volume (fs_public.h)
FS_CREATE_OSDATA = 16

#
# System Storage supported endpoints - Visible to all ESX services
#
BOOTBANK_LINK = os.path.join(os.path.sep, 'bootbank')
ALTBOOTBANK_LINK = os.path.join(os.path.sep, 'altbootbank')
OSDATA_LINK = os.path.join(os.path.sep, 'var', 'lib', 'vmware', 'osdata')
VARCACHE_LINK = os.path.join(os.path.sep, 'var', 'cache')
VARLOG_LINK = os.path.join(os.path.sep, 'var', 'log')
VARCORE_LINK = os.path.join(os.path.sep, 'var', 'core')

#
# Deprecated - Legacy, please do not use.
#
SCRATCH_LINK = os.path.join(os.path.sep, 'scratch')
LOCKER_LINK = os.path.join(os.path.sep, 'locker')
STORE_LINK = os.path.join(os.path.sep, 'store')

#
# VMK Boot options
#
# autoPartitionOSDataSize option is for internal testing purposes only.
# It must not be exposed outside of VMware.
AUTOPART_OSDATA_SIZE_OPT = 'autoPartitionOSDataSize'

# Media size specifier when in the installer
SYSTEM_MEDIA_SIZE_OPT = 'systemMediaSize'

